<?php

/**
 * StatusPage Block.
 *
 * Registers the Gutenberg block for status widgets.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Block
 *
 * Handles Gutenberg block registration and rendering.
 */
class StatusPage_Block
{

    /**
     * Initialize the block.
     */
    public static function init()
    {
        // Register block directly - don't use add_action since we're already in init
        self::register_block();
        add_action('enqueue_block_editor_assets', array(__CLASS__, 'enqueue_editor_assets'));
    }

    /**
     * Register the block.
     */
    public static function register_block()
    {
        // Block directory path - wp-scripts builds to build/blocks/status-widget/
        $block_dir = STATUSPAGE_WIDGET_PATH . 'build/blocks/status-widget';

        // Check if build directory exists
        if (! file_exists($block_dir . '/block.json')) {
            // Fallback: try legacy build path
            $block_dir = STATUSPAGE_WIDGET_PATH . 'build';
        }

        // Register the block using block.json
        register_block_type(
            $block_dir,
            array(
                'render_callback' => array(__CLASS__, 'render_block'),
            )
        );
    }

    /**
     * Enqueue block editor assets.
     */
    public static function enqueue_editor_assets()
    {
        // Get defaults from settings.
        $defaults = StatusPage_Settings::get_defaults();

        // The script handle is auto-generated from block.json as: statuspage-widget-status-widget-editor-script
        wp_localize_script(
            'statuspage-widget-status-widget-editor-script',
            'statuspageWidgetData',
            array(
                'apiBase'   => STATUSPAGE_API_BASE,
                'restNonce' => wp_create_nonce('wp_rest'),
                'restUrl'   => rest_url('statuspage-widget/v1/'),
                'defaults'  => array(
                    'slug'          => $defaults['slug'],
                    'theme'         => $defaults['theme'],
                    'compact'       => $defaults['compact'],
                    'showIncidents' => $defaults['show_incidents'],
                    'rtm'           => $defaults['rtm'],
                    'upd'           => $defaults['upd'],
                    'refresh'       => $defaults['refresh'] * 1000, // Convert to milliseconds
                    'disableLink'   => $defaults['disable_link'],
                    'renderMode'    => $defaults['render_mode'],
                ),
            )
        );
    }

    /**
     * Render the block on the frontend.
     *
     * @param array  $attributes Block attributes.
     * @param string $content    Block content.
     * @return string Rendered HTML.
     */
    public static function render_block($attributes, $content)
    {
        $render_atts = array(
            'slug'           => isset($attributes['slug']) ? $attributes['slug'] : '',
            'theme'          => isset($attributes['theme']) ? $attributes['theme'] : 'auto',
            'compact'        => isset($attributes['compact']) ? $attributes['compact'] : false,
            'show_incidents' => isset($attributes['showIncidents']) ? $attributes['showIncidents'] : false,
            'rtm'            => isset($attributes['rtm']) ? $attributes['rtm'] : 60,
            'upd'            => isset($attributes['upd']) ? $attributes['upd'] : 7,
            'disable_link'   => isset($attributes['disableLink']) ? $attributes['disableLink'] : false,
            'render_mode'    => isset($attributes['renderMode']) ? $attributes['renderMode'] : 'js',
        );

        // Add wrapper with block-specific classes.
        $wrapper_attributes = get_block_wrapper_attributes();

        return sprintf(
            '<div %s>%s</div>',
            $wrapper_attributes,
            StatusPage_Renderer::render($render_atts)
        );
    }
}
